/*******************************************************************************
* Copyright (c) 2015 Jeff Martin.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the GNU Lesser General Public
* License v3.0 which accompanies this distribution, and is available at
* http://www.gnu.org/licenses/lgpl.html
*
* <p>Contributors:
* Jeff Martin - initial API and implementation
******************************************************************************/

package cuchaz.enigma.translation.representation.entry;

import java.util.Arrays;
import java.util.Objects;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import cuchaz.enigma.api.view.entry.ClassDefEntryView;
import cuchaz.enigma.source.RenamableTokenType;
import cuchaz.enigma.translation.TranslateResult;
import cuchaz.enigma.translation.Translator;
import cuchaz.enigma.translation.mapping.EntryMapping;
import cuchaz.enigma.translation.representation.AccessFlags;
import cuchaz.enigma.translation.representation.Signature;

public class ClassDefEntry extends ClassEntry implements DefEntry<ClassEntry>, ClassDefEntryView {
	private final AccessFlags access;
	private final Signature signature;
	private final @Nullable ClassEntry superClass;
	private final ClassEntry[] interfaces;

	public ClassDefEntry(String className, Signature signature, AccessFlags access, @Nullable ClassEntry superClass, ClassEntry[] interfaces) {
		this(getOuterClass(className), getInnerName(className), signature, access, superClass, interfaces, null);
	}

	public ClassDefEntry(ClassEntry parent, String className, Signature signature, AccessFlags access, @Nullable ClassEntry superClass, ClassEntry[] interfaces) {
		this(parent, className, signature, access, superClass, interfaces, null);
	}

	public ClassDefEntry(ClassEntry parent, String className, Signature signature, AccessFlags access, @Nullable ClassEntry superClass, ClassEntry[] interfaces, String javadocs) {
		super(parent, className, javadocs);

		this.signature = Objects.requireNonNull(signature, "Class signature cannot be null");
		this.access = Objects.requireNonNull(access, "Class access cannot be null");
		this.superClass = superClass;
		this.interfaces = interfaces != null ? interfaces : new ClassEntry[0];
	}

	public static ClassDefEntry parse(int access, String name, String signature, String superName, String[] interfaces) {
		ClassEntry superClass = superName != null ? new ClassEntry(superName) : null;
		ClassEntry[] interfaceClasses = Arrays.stream(interfaces).map(ClassEntry::new).toArray(ClassEntry[]::new);
		return new ClassDefEntry(name, Signature.createSignature(signature), new AccessFlags(access), superClass, interfaceClasses);
	}

	public Signature getSignature() {
		return signature;
	}

	@Override
	public AccessFlags getAccess() {
		return access;
	}

	@Override
	@Nullable
	public ClassEntry getSuperClass() {
		return superClass;
	}

	@Override
	public ClassEntry[] getInterfaces() {
		return interfaces;
	}

	public boolean isRecord() {
		return superClass != null && superClass.getName().equals("java/lang/Record");
	}

	@Override
	public TranslateResult<ClassDefEntry> extendedTranslate(Translator translator, @NotNull EntryMapping mapping) {
		Signature translatedSignature = translator.translate(signature);
		String translatedName = mapping.targetName() != null ? mapping.targetName() : name;
		AccessFlags translatedAccess = mapping.accessModifier().transform(access);
		ClassEntry translatedSuper = translator.translate(superClass);
		ClassEntry[] translatedInterfaces = Arrays.stream(interfaces).map(translator::translate).toArray(ClassEntry[]::new);
		String docs = mapping.javadoc();
		return TranslateResult.of(mapping.targetName() == null ? RenamableTokenType.OBFUSCATED : RenamableTokenType.DEOBFUSCATED, new ClassDefEntry(parent, translatedName, translatedSignature, translatedAccess, translatedSuper, translatedInterfaces, docs));
	}

	@Override
	public ClassDefEntry withName(String name) {
		return new ClassDefEntry(parent, name, signature, access, superClass, interfaces, javadocs);
	}

	@Override
	public ClassDefEntry withParent(ClassEntry parent) {
		return new ClassDefEntry(parent, name, signature, access, superClass, interfaces, javadocs);
	}
}
