/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import net.minecraft.class_1959;
import net.minecraft.class_5321;
import net.minecraft.class_6544;
import net.minecraft.class_8197;
import com.google.common.base.Preconditions;
import com.mojang.datafixers.util.Pair;

/**
 * Internal data for modding Vanilla's {@link class_8197.class_5305#field_24723}.
 */
public final class NetherBiomeData {
	// Cached sets of the biomes that would generate from Vanilla's default biome source without consideration
	// for data packs (as those would be distinct biome sources).
	private static final Set<class_5321<class_1959>> NETHER_BIOMES = new HashSet<>();

	private static final Map<class_5321<class_1959>, class_6544.class_4762> NETHER_BIOME_NOISE_POINTS = new HashMap<>();

	private NetherBiomeData() {
	}

	public static void addNetherBiome(class_5321<class_1959> biome, class_6544.class_4762 spawnNoisePoint) {
		Preconditions.checkArgument(biome != null, "Biome is null");
		Preconditions.checkArgument(spawnNoisePoint != null, "MultiNoiseUtil.NoiseValuePoint is null");
		NETHER_BIOME_NOISE_POINTS.put(biome, spawnNoisePoint);
		clearBiomeSourceCache();
	}

	public static boolean canGenerateInNether(class_5321<class_1959> biome) {
		return class_8197.class_5305.field_24723.method_49514().anyMatch(input -> input.equals(biome));
	}

	private static void clearBiomeSourceCache() {
		NETHER_BIOMES.clear(); // Clear cached biome source data
	}

	public static <T> class_6544.class_6547<T> withModdedBiomeEntries(class_6544.class_6547<T> entries, Function<class_5321<class_1959>, T> biomes) {
		if (NETHER_BIOME_NOISE_POINTS.isEmpty()) {
			return entries;
		}

		ArrayList<Pair<class_6544.class_4762, T>> entryList = new ArrayList<>(entries.method_38128());

		for (Map.Entry<class_5321<class_1959>, class_6544.class_4762> entry : NETHER_BIOME_NOISE_POINTS.entrySet()) {
			entryList.add(Pair.of(entry.getValue(), biomes.apply(entry.getKey())));
		}

		return new class_6544.class_6547<>(Collections.unmodifiableList(entryList));
	}
}
