/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome.modification;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import net.fabricmc.fabric.api.biome.v1.BiomeModificationContext;
import net.fabricmc.fabric.mixin.biome.modification.BiomeAccessor;
import net.fabricmc.fabric.mixin.biome.modification.BiomeEffectsAccessor;
import net.fabricmc.fabric.mixin.biome.modification.BiomeWeatherAccessor;
import net.fabricmc.fabric.mixin.biome.modification.GenerationSettingsAccessor;
import net.fabricmc.fabric.mixin.biome.modification.SpawnDensityAccessor;
import net.fabricmc.fabric.mixin.biome.modification.SpawnSettingsAccessor;
import net.minecraft.class_1299;
import net.minecraft.class_1311;
import net.minecraft.class_1959;
import net.minecraft.class_2378;
import net.minecraft.class_2893;
import net.minecraft.class_2922;
import net.minecraft.class_2975;
import net.minecraft.class_3031;
import net.minecraft.class_3195;
import net.minecraft.class_3414;
import net.minecraft.class_3504;
import net.minecraft.class_4761;
import net.minecraft.class_4763;
import net.minecraft.class_4967;
import net.minecraft.class_4968;
import net.minecraft.class_5195;
import net.minecraft.class_5312;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_5483;
import net.minecraft.class_5485;

@ApiStatus.Internal
public class BiomeModificationContextImpl implements BiomeModificationContext {
	private final class_5455 registries;
	private final class_1959 biome;
	private final BiomeAccessor biomeAccessor;
	private final WeatherContext weather;
	private final EffectsContext effects;
	private final GenerationSettingsContextImpl generationSettings;
	private final SpawnSettingsContextImpl spawnSettings;

	@SuppressWarnings("ConstantConditions")
	public BiomeModificationContextImpl(class_5455 registries, class_1959 biome) {
		this.registries = registries;
		this.biome = biome;
		this.biomeAccessor = (BiomeAccessor) (Object) biome;
		this.weather = new WeatherContextImpl();
		this.effects = new EffectsContextImpl();
		this.generationSettings = new GenerationSettingsContextImpl();
		this.spawnSettings = new SpawnSettingsContextImpl();
	}

	@Override
	public void setDepth(float depth) {
		biomeAccessor.fabric_setDepth(depth);
	}

	@Override
	public void setScale(float scale) {
		biomeAccessor.fabric_setScale(scale);
	}

	@Override
	public void setCategory(class_1959.class_1961 category) {
		biomeAccessor.fabric_setCategory(category);
	}

	@Override
	public WeatherContext getWeather() {
		return weather;
	}

	@Override
	public EffectsContext getEffects() {
		return effects;
	}

	@Override
	public GenerationSettingsContext getGenerationSettings() {
		return generationSettings;
	}

	@Override
	public SpawnSettingsContext getSpawnSettings() {
		return spawnSettings;
	}

	/**
	 * Re-freeze any immutable lists and perform general post-modification cleanup.
	 */
	void freeze() {
		generationSettings.freeze();
		spawnSettings.freeze();
	}

	private class WeatherContextImpl implements WeatherContext {
		private final BiomeWeatherAccessor accessor = (BiomeWeatherAccessor) biomeAccessor.fabric_getWeather();

		@Override
		public void setPrecipitation(class_1959.class_1963 precipitation) {
			Objects.requireNonNull(precipitation);
			accessor.setPrecipitation(precipitation);
		}

		@Override
		public void setTemperature(float temperature) {
			accessor.setTemperature(temperature);
		}

		@Override
		public void setTemperatureModifier(class_1959.class_5484 temperatureModifier) {
			Objects.requireNonNull(temperatureModifier);
			accessor.setTemperatureModifier(temperatureModifier);
		}

		@Override
		public void setDownfall(float downfall) {
			accessor.setDownfall(downfall);
		}
	}

	private class EffectsContextImpl implements EffectsContext {
		private final BiomeEffectsAccessor accessor = (BiomeEffectsAccessor) biome.method_24377();

		@Override
		public void setFogColor(int color) {
			accessor.fabric_setFogColor(color);
		}

		@Override
		public void setWaterColor(int color) {
			accessor.fabric_setWaterColor(color);
		}

		@Override
		public void setWaterFogColor(int color) {
			accessor.fabric_setWaterFogColor(color);
		}

		@Override
		public void setSkyColor(int color) {
			accessor.fabric_setSkyColor(color);
		}

		@Override
		public void setFoliageColor(Optional<Integer> color) {
			Objects.requireNonNull(color);
			accessor.fabric_setFoliageColor(color);
		}

		@Override
		public void setGrassColor(Optional<Integer> color) {
			Objects.requireNonNull(color);
			accessor.fabric_setGrassColor(color);
		}

		@Override
		public void setGrassColorModifier(@NotNull class_4763.class_5486 colorModifier) {
			Objects.requireNonNull(colorModifier);
			accessor.fabric_setGrassColorModifier(colorModifier);
		}

		@Override
		public void setParticleConfig(Optional<class_4761> particleConfig) {
			Objects.requireNonNull(particleConfig);
			accessor.fabric_setParticleConfig(particleConfig);
		}

		@Override
		public void setAmbientSound(Optional<class_3414> sound) {
			Objects.requireNonNull(sound);
			accessor.fabric_setLoopSound(sound);
		}

		@Override
		public void setMoodSound(Optional<class_4968> sound) {
			Objects.requireNonNull(sound);
			accessor.fabric_setMoodSound(sound);
		}

		@Override
		public void setAdditionsSound(Optional<class_4967> sound) {
			Objects.requireNonNull(sound);
			accessor.fabric_setAdditionsSound(sound);
		}

		@Override
		public void setMusic(Optional<class_5195> sound) {
			Objects.requireNonNull(sound);
			accessor.fabric_setMusic(sound);
		}
	}

	private class GenerationSettingsContextImpl implements GenerationSettingsContext {
		private final class_2378<class_2922<?>> carvers = registries.method_30530(class_2378.field_25913);
		private final class_2378<class_2975<?, ?>> features = registries.method_30530(class_2378.field_25914);
		private final class_2378<class_5312<?, ?>> structures = registries.method_30530(class_2378.field_25915);
		private final class_2378<class_3504<?>> surfaceBuilders = registries.method_30530(class_2378.field_25912);
		private final class_5485 generationSettings = biome.method_30970();
		private final GenerationSettingsAccessor accessor = (GenerationSettingsAccessor) generationSettings;

		/**
		 * Unfreeze the immutable lists found in the generation settings, and make sure they're filled up to every
		 * possible step if they're dense lists.
		 */
		GenerationSettingsContextImpl() {
			unfreezeCarvers();
			unfreezeFeatures();
			unfreezeFlowerFeatures();
			unfreezeStructures();
		}

		private void unfreezeCarvers() {
			Map<class_2893.class_2894, List<Supplier<class_2922<?>>>> carversByStep = new EnumMap<>(class_2893.class_2894.class);
			carversByStep.putAll(accessor.fabric_getCarvers());

			for (class_2893.class_2894 step : class_2893.class_2894.values()) {
				List<Supplier<class_2922<?>>> carvers = carversByStep.get(step);

				if (carvers == null) {
					carvers = new ArrayList<>();
				} else {
					carvers = new ArrayList<>(carvers);
				}

				carversByStep.put(step, carvers);
			}

			accessor.fabric_setCarvers(carversByStep);
		}

		private void unfreezeFeatures() {
			List<List<Supplier<class_2975<?, ?>>>> features = accessor.fabric_getFeatures();
			features = new ArrayList<>(features);

			for (int i = 0; i < features.size(); i++) {
				features.set(i, new ArrayList<>(features.get(i)));
			}

			accessor.fabric_setFeatures(features);
		}

		private void unfreezeFlowerFeatures() {
			accessor.fabric_setFlowerFeatures(new ArrayList<>(accessor.fabric_getFlowerFeatures()));
		}

		private void unfreezeStructures() {
			accessor.fabric_setStructureFeatures(new ArrayList<>(accessor.fabric_getStructureFeatures()));
		}

		/**
		 * Re-freeze the lists in the generation settings to immutable variants, also fixes the flower features.
		 */
		public void freeze() {
			freezeCarvers();
			freezeFeatures();
			freezeFlowerFeatures();
			freezeStructures();
		}

		private void freezeCarvers() {
			Map<class_2893.class_2894, List<Supplier<class_2922<?>>>> carversByStep = accessor.fabric_getCarvers();

			for (class_2893.class_2894 step : class_2893.class_2894.values()) {
				carversByStep.put(step, ImmutableList.copyOf(carversByStep.get(step)));
			}

			accessor.fabric_setCarvers(ImmutableMap.copyOf(carversByStep));
		}

		private void freezeFeatures() {
			List<List<Supplier<class_2975<?, ?>>>> featureSteps = accessor.fabric_getFeatures();

			for (int i = 0; i < featureSteps.size(); i++) {
				featureSteps.set(i, ImmutableList.copyOf(featureSteps.get(i)));
			}

			accessor.fabric_setFeatures(ImmutableList.copyOf(featureSteps));
		}

		private void freezeFlowerFeatures() {
			accessor.fabric_setFlowerFeatures(ImmutableList.copyOf(accessor.fabric_getFlowerFeatures()));
		}

		private void freezeStructures() {
			accessor.fabric_setStructureFeatures(ImmutableList.copyOf(accessor.fabric_getStructureFeatures()));
		}

		@Override
		public void setSurfaceBuilder(class_5321<class_3504<?>> surfaceBuilderKey) {
			// We do not need to delay evaluation of this since the registries are already fully built
			class_3504<?> surfaceBuilder = surfaceBuilders.method_31140(surfaceBuilderKey);
			accessor.fabric_setSurfaceBuilder(() -> surfaceBuilder);
		}

		@Override
		public boolean removeFeature(class_2893.class_2895 step, class_5321<class_2975<?, ?>> configuredFeatureKey) {
			class_2975<?, ?> configuredFeature = features.method_31140(configuredFeatureKey);

			int stepIndex = step.ordinal();
			List<List<Supplier<class_2975<?, ?>>>> featureSteps = accessor.fabric_getFeatures();

			if (stepIndex >= featureSteps.size()) {
				return false; // The step was not populated with any features yet
			}

			List<Supplier<class_2975<?, ?>>> featuresInStep = featureSteps.get(stepIndex);

			if (featuresInStep.removeIf(supplier -> supplier.get() == configuredFeature)) {
				rebuildFlowerFeatures();
				return true;
			}

			return false;
		}

		@Override
		public void addFeature(class_2893.class_2895 step, class_5321<class_2975<?, ?>> configuredFeatureKey) {
			// We do not need to delay evaluation of this since the registries are already fully built
			class_2975<?, ?> configuredFeature = features.method_31140(configuredFeatureKey);

			List<List<Supplier<class_2975<?, ?>>>> featureSteps = accessor.fabric_getFeatures();
			int index = step.ordinal();

			// Add new empty lists for the generation steps that have no features yet
			while (index >= featureSteps.size()) {
				featureSteps.add(new ArrayList<>());
			}

			featureSteps.get(index).add(() -> configuredFeature);

			// Ensure the list of flower features is up to date
			rebuildFlowerFeatures();
		}

		@Override
		public void addCarver(class_2893.class_2894 step, class_5321<class_2922<?>> carverKey) {
			// We do not need to delay evaluation of this since the registries are already fully built
			class_2922<?> carver = carvers.method_31140(carverKey);
			accessor.fabric_getCarvers().get(step).add(() -> carver);
		}

		@Override
		public boolean removeCarver(class_2893.class_2894 step, class_5321<class_2922<?>> configuredCarverKey) {
			class_2922<?> carver = carvers.method_31140(configuredCarverKey);
			return accessor.fabric_getCarvers().get(step).removeIf(supplier -> supplier.get() == carver);
		}

		@Override
		public void addStructure(class_5321<class_5312<?, ?>> configuredStructureKey) {
			class_5312<?, ?> configuredStructure = structures.method_31140(configuredStructureKey);

			// Remove the same feature-type before adding it back again, i.e. a jungle and normal village
			// are mutually exclusive.
			removeStructure(configuredStructure.field_24835);

			accessor.fabric_getStructureFeatures().add(() -> configuredStructure);
		}

		@Override
		public boolean removeStructure(class_5321<class_5312<?, ?>> configuredStructureKey) {
			class_5312<?, ?> structure = structures.method_31140(configuredStructureKey);

			return accessor.fabric_getStructureFeatures().removeIf(s -> s.get() == structure);
		}

		@Override
		public boolean removeStructure(class_3195<?> structure) {
			return accessor.fabric_getStructureFeatures().removeIf(s -> s.get().field_24835 == structure);
		}

		/**
		 * See the constructor of {@link class_5485} for reference.
		 */
		private void rebuildFlowerFeatures() {
			List<class_2975<?, ?>> flowerFeatures = accessor.fabric_getFlowerFeatures();
			flowerFeatures.clear();

			for (List<Supplier<class_2975<?, ?>>> features : accessor.fabric_getFeatures()) {
				for (Supplier<class_2975<?, ?>> supplier : features) {
					supplier.get().method_30648()
							.filter(configuredFeature -> configuredFeature.field_13376 == class_3031.field_21219)
							.forEachOrdered(flowerFeatures::add);
				}
			}
		}
	}

	private class SpawnSettingsContextImpl implements SpawnSettingsContext {
		private final SpawnSettingsAccessor accessor = (SpawnSettingsAccessor) biome.method_30966();

		SpawnSettingsContextImpl() {
			unfreezeSpawners();
			unfreezeSpawnCost();
		}

		private void unfreezeSpawners() {
			EnumMap<class_1311, List<class_5483.class_1964>> spawners = new EnumMap<>(class_1311.class);
			spawners.putAll(accessor.fabric_getSpawners());

			for (class_1311 spawnGroup : class_1311.values()) {
				List<class_5483.class_1964> entries = spawners.get(spawnGroup);

				if (entries != null) {
					spawners.put(spawnGroup, new ArrayList<>(entries));
				} else {
					spawners.put(spawnGroup, new ArrayList<>());
				}
			}

			accessor.fabric_setSpawners(spawners);
		}

		private void unfreezeSpawnCost() {
			accessor.fabric_setSpawnCosts(new HashMap<>(accessor.fabric_getSpawnCosts()));
		}

		public void freeze() {
			freezeSpawners();
			freezeSpawnCosts();
		}

		private void freezeSpawners() {
			Map<class_1311, List<class_5483.class_1964>> spawners = accessor.fabric_getSpawners();

			for (Map.Entry<class_1311, List<class_5483.class_1964>> entry : spawners.entrySet()) {
				entry.setValue(ImmutableList.copyOf(entry.getValue()));
			}

			accessor.fabric_setSpawners(ImmutableMap.copyOf(spawners));
		}

		private void freezeSpawnCosts() {
			accessor.fabric_setSpawnCosts(ImmutableMap.copyOf(accessor.fabric_getSpawnCosts()));
		}

		@Override
		public void setPlayerSpawnFriendly(boolean playerSpawnFriendly) {
			accessor.fabric_setPlayerSpawnFriendly(playerSpawnFriendly);
		}

		@Override
		public void setCreatureSpawnProbability(float probability) {
			accessor.fabric_setCreatureSpawnProbability(probability);
		}

		@Override
		public void addSpawn(class_1311 spawnGroup, class_5483.class_1964 spawnEntry) {
			Objects.requireNonNull(spawnGroup);
			Objects.requireNonNull(spawnEntry);

			accessor.fabric_getSpawners().get(spawnGroup).add(spawnEntry);
		}

		@Override
		public boolean removeSpawns(BiPredicate<class_1311, class_5483.class_1964> predicate) {
			Map<class_1311, List<class_5483.class_1964>> spawners = accessor.fabric_getSpawners();
			boolean anyRemoved = false;

			for (class_1311 group : class_1311.values()) {
				if (spawners.get(group).removeIf(entry -> predicate.test(group, entry))) {
					anyRemoved = true;
				}
			}

			return anyRemoved;
		}

		@Override
		public void setSpawnCost(class_1299<?> entityType, double mass, double gravityLimit) {
			Objects.requireNonNull(entityType);
			accessor.fabric_getSpawnCosts().put(entityType, SpawnDensityAccessor.create(gravityLimit, mass));
		}

		@Override
		public void clearSpawnCost(class_1299<?> entityType) {
			accessor.fabric_getSpawnCosts().remove(entityType);
		}
	}
}
