/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.container;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import io.netty.buffer.Unpooled;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.api.container.ContainerFactory;
import net.fabricmc.fabric.api.container.ContainerProviderRegistry;
import net.fabricmc.fabric.impl.networking.PacketTypes;
import net.fabricmc.fabric.mixin.container.ServerPlayerEntityAccessor;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_2540;
import net.minecraft.class_2658;
import net.minecraft.class_2960;
import net.minecraft.class_3222;

public class ContainerProviderImpl implements ContainerProviderRegistry {
	/**
	 * Use the instance provided by ContainerProviderRegistry.
	 */
	public static final ContainerProviderImpl INSTANCE = new ContainerProviderImpl();

	private static final Logger LOGGER = LogManager.getLogger();

	private static final Map<class_2960, ContainerFactory<class_1703>> FACTORIES = new HashMap<>();

	@Override
	public void registerFactory(class_2960 identifier, ContainerFactory<class_1703> factory) {
		if (FACTORIES.containsKey(identifier)) {
			throw new RuntimeException("A factory has already been registered as " + identifier.toString());
		}

		FACTORIES.put(identifier, factory);
	}

	@Override
	public void openContainer(class_2960 identifier, class_1657 player, Consumer<class_2540> writer) {
		if (!(player instanceof class_3222)) {
			LOGGER.warn("Please only use ContainerProviderRegistry.openContainer() with server-sided player entities!");
			return;
		}

		openContainer(identifier, (class_3222) player, writer);
	}

	private boolean emittedNoSyncHookWarning = false;

	@Override
	public void openContainer(class_2960 identifier, class_3222 player, Consumer<class_2540> writer) {
		int syncId;

		if (player instanceof ServerPlayerEntitySyncHook) {
			ServerPlayerEntitySyncHook serverPlayerEntitySyncHook = (ServerPlayerEntitySyncHook) player;
			syncId = serverPlayerEntitySyncHook.fabric_incrementSyncId();
		} else if (player instanceof ServerPlayerEntityAccessor) {
			if (!emittedNoSyncHookWarning) {
				LOGGER.warn("ServerPlayerEntitySyncHook could not be applied - fabric-containers is using a hack!");
				emittedNoSyncHookWarning = true;
			}

			syncId = (((ServerPlayerEntityAccessor) player).getScreenHandlerSyncId() + 1) % 100;
			((ServerPlayerEntityAccessor) player).setScreenHandlerSyncId(syncId);
		} else {
			throw new RuntimeException("Neither ServerPlayerEntitySyncHook nor Accessor present! This should not happen!");
		}

		class_2540 buf = new class_2540(Unpooled.buffer());
		buf.method_10812(identifier);
		buf.writeByte(syncId);

		writer.accept(buf);
		player.field_13987.method_14364(new class_2658(PacketTypes.OPEN_CONTAINER, buf));

		class_2540 clonedBuf = new class_2540(buf.duplicate());
		clonedBuf.method_10810();
		clonedBuf.readUnsignedByte();

		class_1703 container = createContainer(syncId, identifier, player, clonedBuf);

		if (container == null) {
			return;
		}

		player.field_7512 = container;
		player.field_7512.method_7596(player);
	}

	public <C extends class_1703> C createContainer(int syncId, class_2960 identifier, class_1657 player, class_2540 buf) {
		ContainerFactory<class_1703> factory = FACTORIES.get(identifier);

		if (factory == null) {
			LOGGER.error("No container factory found for {}!", identifier.toString());
			return null;
		}

		//noinspection unchecked
		return (C) factory.create(syncId, identifier, player, buf);
	}
}
