/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.registry;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.mixin.content.registry.AxeItemAccessor;
import net.minecraft.class_2248;
import net.minecraft.class_2741;

/**
 * A registry for axe stripping interactions. A vanilla example is turning logs to stripped logs.
 */
public final class StrippableBlockRegistry {
	private static final Logger LOGGER = LogManager.getLogger();

	private StrippableBlockRegistry() {
	}

	/**
	 * Registers a stripping interaction.
	 *
	 * <p>Both blocks must have the {@link class_2741#field_12496 axis} property.
	 *
	 * @param input    the input block that can be stripped
	 * @param stripped the stripped result block
	 * @throws IllegalArgumentException if the input or the output doesn't have the {@link class_2741#field_12496 axis} property
	 */
	public static void register(class_2248 input, class_2248 stripped) {
		requireNonNullAndAxisProperty(input, "input block");
		requireNonNullAndAxisProperty(stripped, "stripped block");
		Map<class_2248, class_2248> strippedBlocks = AxeItemAccessor.getStrippedBlocks();

		if (!(strippedBlocks instanceof HashMap<?, ?>)) {
			strippedBlocks = new HashMap<>(strippedBlocks);
			AxeItemAccessor.setStrippedBlocks(strippedBlocks);
		}

		class_2248 old = strippedBlocks.put(input, stripped);

		if (old != null) {
			LOGGER.debug("Replaced old stripping mapping from {} to {} with {}", input, old, stripped);
		}
	}

	private static void requireNonNullAndAxisProperty(class_2248 block, String name) {
		Objects.requireNonNull(block, name + " cannot be null");

		if (!block.method_9595().method_11659().contains(class_2741.field_12496)) {
			throw new IllegalArgumentException(name + " must have the 'axis' property");
		}
	}
}
