/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.attachment.sync;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import net.minecraft.network.Connection;
import net.minecraft.network.protocol.Packet;
import net.minecraft.resources.Identifier;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.network.ConfigurationTask;

import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.entity.event.v1.ServerEntityLevelChangeEvents;
import net.fabricmc.fabric.api.entity.event.v1.ServerPlayerEvents;
import net.fabricmc.fabric.api.networking.v1.EntityTrackingEvents;
import net.fabricmc.fabric.api.networking.v1.PayloadTypeRegistry;
import net.fabricmc.fabric.api.networking.v1.ServerConfigurationConnectionEvents;
import net.fabricmc.fabric.api.networking.v1.ServerConfigurationNetworking;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.fabricmc.fabric.impl.attachment.AttachmentEntrypoint;
import net.fabricmc.fabric.impl.attachment.AttachmentRegistryImpl;
import net.fabricmc.fabric.impl.attachment.AttachmentTargetImpl;
import net.fabricmc.fabric.impl.attachment.sync.clientbound.ClientboundAttachmentSyncPayload;
import net.fabricmc.fabric.impl.attachment.sync.clientbound.ClientboundRequestAcceptedAttachmentsPayload;
import net.fabricmc.fabric.impl.attachment.sync.serverbound.ServerboundAcceptedAttachmentsPayload;
import net.fabricmc.fabric.mixin.networking.accessor.ServerCommonPacketListenerImplAccessor;

public class AttachmentSync implements ModInitializer {
	public static final int MAX_IDENTIFIER_SIZE = 256;

	public static ServerboundAcceptedAttachmentsPayload createResponsePayload() {
		return new ServerboundAcceptedAttachmentsPayload(AttachmentRegistryImpl.getSyncableAttachments());
	}

	public static void trySync(AttachmentChange change, ServerPlayer player) {
		Set<Identifier> supported = ((SupportedAttachmentsConnection) ((ServerCommonPacketListenerImplAccessor) player.connection).getConnection())
				.fabric_getSupportedAttachments();

		if (supported.contains(change.type().identifier())) {
			ServerPlayNetworking.send(player, new ClientboundAttachmentSyncPayload(List.of(change)));
		}
	}

	private static Set<Identifier> decodeResponsePayload(
			ServerboundAcceptedAttachmentsPayload payload) {
		Set<Identifier> atts = payload.acceptedAttachments();
		Set<Identifier> syncable = AttachmentRegistryImpl.getSyncableAttachments();
		atts.retainAll(syncable);

		if (atts.size() < syncable.size()) {
			// Client doesn't support all
			AttachmentEntrypoint.LOGGER.warn(
					"Client does not support the syncable attachments {}",
					syncable.stream().filter(id -> !atts.contains(id)).map(Identifier::toString).collect(Collectors.joining(", "))
			);
		}

		return atts;
	}

	@Override
	public void onInitialize() {
		// Config
		PayloadTypeRegistry.serverboundConfiguration()
				.register(ServerboundAcceptedAttachmentsPayload.ID, ServerboundAcceptedAttachmentsPayload.CODEC);
		PayloadTypeRegistry.clientboundConfiguration()
				.register(ClientboundRequestAcceptedAttachmentsPayload.ID, ClientboundRequestAcceptedAttachmentsPayload.CODEC);

		ServerConfigurationConnectionEvents.CONFIGURE.register((handler, server) -> {
			if (ServerConfigurationNetworking.canSend(handler, ClientboundRequestAcceptedAttachmentsPayload.PACKET_ID)) {
				handler.addTask(new AttachmentSyncTask());
			} else {
				AttachmentEntrypoint.LOGGER.debug(
						"Couldn't send attachment configuration packet to client, as the client cannot receive the payload."
				);
			}
		});

		ServerConfigurationNetworking.registerGlobalReceiver(
				ServerboundAcceptedAttachmentsPayload.ID, (payload, context) -> {
					Set<Identifier> supportedAttachments = decodeResponsePayload(payload);
					Connection connection = ((ServerCommonPacketListenerImplAccessor) context.packetListener()).getConnection();
					((SupportedAttachmentsConnection) connection).fabric_setSupportedAttachments(supportedAttachments);

					context.packetListener().completeTask(AttachmentSyncTask.KEY);
				});

		// Play
		PayloadTypeRegistry.clientboundPlay().register(
				ClientboundAttachmentSyncPayload.ID, ClientboundAttachmentSyncPayload.CODEC);

		ServerPlayerEvents.JOIN.register((player) -> {
			List<AttachmentChange> changes = new ArrayList<>();
			// sync level attachments
			((AttachmentTargetImpl) player.level()).fabric_computeInitialSyncChanges(player, changes::add);
			// sync player's own persistent attachments that couldn't be synced earlier
			((AttachmentTargetImpl) player).fabric_computeInitialSyncChanges(player, changes::add);

			if (!changes.isEmpty()) {
				AttachmentChange.partitionAndSendPackets(changes, player);
			}
		});

		ServerEntityLevelChangeEvents.AFTER_PLAYER_CHANGE_LEVEL.register((player, origin, destination) -> {
			// sync new level's attachments
			// no conflict with previous one because the client level is recreated every time
			List<AttachmentChange> changes = new ArrayList<>();
			((AttachmentTargetImpl) destination).fabric_computeInitialSyncChanges(player, changes::add);

			if (!changes.isEmpty()) {
				AttachmentChange.partitionAndSendPackets(changes, player);
			}
		});

		EntityTrackingEvents.START_TRACKING.register((trackedEntity, player) -> {
			List<AttachmentChange> changes = new ArrayList<>();
			((AttachmentTargetImpl) trackedEntity).fabric_computeInitialSyncChanges(player, changes::add);

			if (!changes.isEmpty()) {
				AttachmentChange.partitionAndSendPackets(changes, player);
			}
		});
	}

	private record AttachmentSyncTask() implements ConfigurationTask {
		public static final Type KEY = new Type(
				ClientboundRequestAcceptedAttachmentsPayload.PACKET_ID.toString());

		@Override
		public void start(Consumer<Packet<?>> sender) {
			sender.accept(ServerConfigurationNetworking.createClientboundPacket(
					ClientboundRequestAcceptedAttachmentsPayload.INSTANCE));
		}

		@Override
		public Type type() {
			return KEY;
		}
	}
}
