/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

import com.google.gson.JsonElement;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.JsonOps;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.minecraft.class_2378;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6903;
import net.minecraft.class_7225;
import net.minecraft.class_7403;
import net.minecraft.class_7784;

/**
 * Extend this class and implement {@link FabricCodecDataProvider#configure(BiConsumer, class_7225.class_7874)}.
 *
 * <p>Register an instance of the class with {@link FabricDataGenerator.Pack#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}.
 */
public abstract class FabricCodecDataProvider<T> implements class_2405 {
	private final class_7784.class_7489 pathResolver;
	private final CompletableFuture<class_7225.class_7874> registriesFuture;
	private final Codec<T> codec;

	private FabricCodecDataProvider(class_7784.class_7489 pathResolver, CompletableFuture<class_7225.class_7874> registriesFuture, Codec<T> codec) {
		this.pathResolver = pathResolver;
		this.registriesFuture = Objects.requireNonNull(registriesFuture);
		this.codec = codec;
	}

	protected FabricCodecDataProvider(FabricDataOutput dataOutput, CompletableFuture<class_7225.class_7874> registriesFuture, class_7784.class_7490 outputType, String directoryName, Codec<T> codec) {
		this(dataOutput.method_45973(outputType, directoryName), registriesFuture, codec);
	}

	protected FabricCodecDataProvider(FabricDataOutput dataOutput, CompletableFuture<class_7225.class_7874> registriesFuture, class_5321<? extends class_2378<?>> key, Codec<T> codec) {
		this(dataOutput.method_60917(key), registriesFuture, codec);
	}

	@Override
	public CompletableFuture<?> method_10319(class_7403 writer) {
		return this.registriesFuture.thenCompose(lookup -> {
			Map<class_2960, JsonElement> entries = new HashMap<>();
			class_6903<JsonElement> ops = lookup.method_57093(JsonOps.INSTANCE);

			BiConsumer<class_2960, T> provider = (id, value) -> {
				JsonElement json = this.convert(id, value, ops);
				JsonElement existingJson = entries.put(id, json);

				if (existingJson != null) {
					throw new IllegalArgumentException("Duplicate entry " + id);
				}
			};

			this.configure(provider, lookup);
			return this.write(writer, entries);
		});
	}

	/**
	 * Implement this method to register entries to generate using a {@link class_7225.class_7874}.
	 * @param provider A consumer that accepts an {@link class_2960} and a value to register.
	 * @param lookup A lookup for registries.
	 */
	protected abstract void configure(BiConsumer<class_2960, T> provider, class_7225.class_7874 lookup);

	private JsonElement convert(class_2960 id, T value, DynamicOps<JsonElement> ops) {
		DataResult<JsonElement> dataResult = this.codec.encodeStart(ops, value);
		return dataResult
				.mapError(message -> "Invalid entry %s: %s".formatted(id, message))
				.getOrThrow();
	}

	private CompletableFuture<?> write(class_7403 writer, Map<class_2960, JsonElement> entries) {
		return CompletableFuture.allOf(entries.entrySet().stream().map(entry -> {
			Path path = this.pathResolver.method_44107(entry.getKey());
			return class_2405.method_10320(writer, entry.getValue(), path);
		}).toArray(CompletableFuture[]::new));
	}
}
