/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.serialization.JsonOps;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.datagen.v1.FabricDataOutput;
import net.fabricmc.fabric.api.resource.conditions.v1.ResourceCondition;
import net.fabricmc.fabric.impl.datagen.FabricDataGenHelper;
import net.minecraft.class_161;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_6903;
import net.minecraft.class_7225;
import net.minecraft.class_7403;
import net.minecraft.class_7784;
import net.minecraft.class_7924;
import net.minecraft.class_8779;

/**
 * Extend this class and implement {@link FabricAdvancementProvider#generateAdvancement}.
 *
 * <p>Register an instance of the class with {@link FabricDataGenerator.Pack#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}.
 */
public abstract class FabricAdvancementProvider implements class_2405 {
	protected final FabricDataOutput output;
	private final class_7784.class_7489 pathResolver;
	private final CompletableFuture<class_7225.class_7874> registryLookup;

	protected FabricAdvancementProvider(FabricDataOutput output, CompletableFuture<class_7225.class_7874> registryLookup) {
		this.output = output;
		this.pathResolver = output.method_60917(class_7924.field_52177);
		this.registryLookup = registryLookup;
	}

	/**
	 * Implement this method to register advancements to generate use the consumer callback to register advancements.
	 *
	 * <p>Use {@link class_161.class_162#method_694(Consumer, String)} to help build advancements.
	 */
	public abstract void generateAdvancement(class_7225.class_7874 registryLookup, Consumer<class_8779> consumer);

	/**
	 * Return a new exporter that applies the specified conditions to any advancement it receives.
	 */
	protected Consumer<class_8779> withConditions(Consumer<class_8779> exporter, ResourceCondition... conditions) {
		Preconditions.checkArgument(conditions.length > 0, "Must add at least one condition.");
		return advancement -> {
			FabricDataGenHelper.addConditions(advancement, conditions);
			exporter.accept(advancement);
		};
	}

	@Override
	public CompletableFuture<?> method_10319(class_7403 writer) {
		return this.registryLookup.thenCompose(lookup -> {
			final Set<class_2960> identifiers = Sets.newHashSet();
			final Set<class_8779> advancements = Sets.newHashSet();

			generateAdvancement(lookup, advancements::add);

			class_6903<JsonElement> ops = lookup.method_57093(JsonOps.INSTANCE);
			final List<CompletableFuture<?>> futures = new ArrayList<>();

			for (class_8779 advancement : advancements) {
				if (!identifiers.add(advancement.comp_1919())) {
					throw new IllegalStateException("Duplicate advancement " + advancement.comp_1919());
				}

				JsonObject advancementJson = class_161.field_47179.encodeStart(ops, advancement.comp_1920()).getOrThrow(IllegalStateException::new).getAsJsonObject();
				FabricDataGenHelper.addConditions(advancementJson, FabricDataGenHelper.consumeConditions(advancement));
				futures.add(class_2405.method_10320(writer, advancementJson, getOutputPath(advancement)));
			}

			return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
		});
	}

	private Path getOutputPath(class_8779 advancement) {
		return pathResolver.method_44107(advancement.comp_1919());
	}

	@Override
	public String method_10321() {
		return "Advancements";
	}
}
