/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.datagen.v1.provider;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Set;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.gson.JsonObject;
import net.fabricmc.fabric.api.datagen.v1.FabricDataGenerator;
import net.fabricmc.fabric.api.resource.conditions.v1.ConditionJsonProvider;
import net.fabricmc.fabric.impl.datagen.FabricDataGenHelper;
import net.minecraft.class_161;
import net.minecraft.class_2403;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_7403;

/**
 * Extend this class and implement {@link FabricAdvancementProvider#generateAdvancement}.
 *
 * <p>Register an instance of the class with {@link FabricDataGenerator#addProvider} in a {@link net.fabricmc.fabric.api.datagen.v1.DataGeneratorEntrypoint}
 */
public abstract class FabricAdvancementProvider implements class_2405 {
	protected final FabricDataGenerator dataGenerator;
	private final class_2403.class_7489 pathResolver;

	protected FabricAdvancementProvider(FabricDataGenerator dataGenerator) {
		this.dataGenerator = dataGenerator;
		this.pathResolver = dataGenerator.method_44106(class_2403.class_7490.field_39367, "advancements");
	}

	/**
	 * Implement this method to register advancements to generate use the consumer callback to register advancements.
	 *
	 * <p>Use {@link class_161.Task#build(Consumer, String)} to help build advancements.
	 */
	public abstract void generateAdvancement(Consumer<class_161> consumer);

	/**
	 * Return a new exporter that applies the specified conditions to any advancement it receives.
	 */
	protected Consumer<class_161> withConditions(Consumer<class_161> exporter, ConditionJsonProvider... conditions) {
		Preconditions.checkArgument(conditions.length > 0, "Must add at least one condition.");
		return advancement -> {
			FabricDataGenHelper.addConditions(advancement, conditions);
			exporter.accept(advancement);
		};
	}

	@Override
	public void method_10319(class_7403 writer) throws IOException {
		final Set<class_2960> identifiers = Sets.newHashSet();
		final Set<class_161> advancements = Sets.newHashSet();

		generateAdvancement(advancements::add);

		for (class_161 advancement : advancements) {
			if (!identifiers.add(advancement.method_688())) {
				throw new IllegalStateException("Duplicate advancement " + advancement.method_688());
			}

			JsonObject advancementJson = advancement.method_689().method_698();
			ConditionJsonProvider.write(advancementJson, FabricDataGenHelper.consumeConditions(advancement));

			class_2405.method_10320(writer, advancementJson, getOutputPath(advancement));
		}
	}

	private Path getOutputPath(class_161 advancement) {
		return dataGenerator.method_10313().resolve("data/%s/advancements/%s.json".formatted(advancement.method_688().method_12836(), advancement.method_688().method_12832()));
	}

	@Override
	public String method_10321() {
		return "Advancements";
	}
}
