/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.networking;

import java.util.Queue;
import java.util.Set;
import java.util.function.Function;

import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import com.llamalad7.mixinextras.injector.wrapoperation.WrapOperation;
import io.netty.buffer.ByteBuf;
import org.jetbrains.annotations.Nullable;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import net.minecraft.class_2535;
import net.minecraft.class_5455;
import net.minecraft.class_8605;
import net.minecraft.class_8609;
import net.minecraft.class_8610;
import net.minecraft.class_8792;
import net.minecraft.class_9129;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.api.networking.v1.FabricServerConfigurationNetworkHandler;
import net.fabricmc.fabric.impl.networking.FabricRegistryByteBuf;
import net.fabricmc.fabric.impl.networking.NetworkHandlerExtensions;
import net.fabricmc.fabric.impl.networking.server.ServerConfigurationNetworkAddon;

// We want to apply a bit earlier than other mods which may not use us in order to prevent refCount issues
@Mixin(value = class_8610.class, priority = 900)
public abstract class ServerConfigurationNetworkHandlerMixin extends class_8609 implements NetworkHandlerExtensions, FabricServerConfigurationNetworkHandler {
	@Shadow
	@Nullable
	private class_8605 currentTask;

	@Shadow
	protected abstract void onTaskFinished(class_8605.class_8606 key);

	@Shadow
	@Final
	private Queue<class_8605> tasks;

	@Shadow
	public abstract boolean method_48106();

	@Shadow
	public abstract void sendConfigurations();

	@Unique
	private ServerConfigurationNetworkAddon addon;

	@Unique
	private boolean sentConfiguration;

	@Unique
	private boolean earlyTaskExecution;

	public ServerConfigurationNetworkHandlerMixin(MinecraftServer server, class_2535 connection, class_8792 arg) {
		super(server, connection, arg);
	}

	@Inject(method = "<init>", at = @At("RETURN"))
	private void initAddon(CallbackInfo ci) {
		this.addon = new ServerConfigurationNetworkAddon((class_8610) (Object) this, this.field_45012);
		// A bit of a hack but it allows the field above to be set in case someone registers handlers during INIT event which refers to said field
		this.addon.lateInit();
	}

	@Inject(method = "sendConfigurations", at = @At("HEAD"), cancellable = true)
	private void onClientReady(CallbackInfo ci) {
		// Send the initial channel registration packet
		if (this.addon.startConfiguration()) {
			if (currentTask != null) {
				throw new IllegalStateException("A task is already running: " + currentTask.method_52375().comp_1576());
			}

			ci.cancel();
			return;
		}

		// Ready to start sending packets
		if (!sentConfiguration) {
			this.addon.preConfiguration();
			sentConfiguration = true;
			earlyTaskExecution = true;
		}

		// Run the early tasks
		if (earlyTaskExecution) {
			if (pollEarlyTasks()) {
				ci.cancel();
				return;
			} else {
				earlyTaskExecution = false;
			}
		}

		// All early tasks should have been completed
		if (currentTask != null || !tasks.isEmpty()) {
			throw new IllegalStateException("All early tasks should have been completed, current: " + currentTask + ", queued: " + tasks.size());
		}

		// Run the vanilla tasks.
		this.addon.configuration();
	}

	@Unique
	private boolean pollEarlyTasks() {
		if (!earlyTaskExecution) {
			throw new IllegalStateException("Early task execution has finished");
		}

		if (this.currentTask != null) {
			throw new IllegalStateException("Task " + this.currentTask.method_52375().comp_1576() + " has not finished yet");
		}

		if (!this.method_48106()) {
			return false;
		}

		final class_8605 task = this.tasks.poll();

		if (task != null) {
			this.currentTask = task;
			task.method_52376(this::method_14364);
			return true;
		}

		return false;
	}

	@Override
	public ServerConfigurationNetworkAddon getAddon() {
		return addon;
	}

	@Override
	public void addTask(class_8605 task) {
		tasks.add(task);
	}

	@Override
	public void completeTask(class_8605.class_8606 key) {
		if (!earlyTaskExecution) {
			onTaskFinished(key);
			return;
		}

		final class_8605.class_8606 currentKey = this.currentTask != null ? this.currentTask.method_52375() : null;

		if (!key.equals(currentKey)) {
			throw new IllegalStateException("Unexpected request for task finish, current task: " + currentKey + ", requested: " + key);
		}

		this.currentTask = null;
		sendConfigurations();
	}

	@WrapOperation(method = "onReady", at = @At(value = "INVOKE", target = "Lnet/minecraft/network/RegistryByteBuf;makeFactory(Lnet/minecraft/registry/DynamicRegistryManager;)Ljava/util/function/Function;"))
	private Function<ByteBuf, class_9129> bindChannelInfo(class_5455 registryManager, Operation<Function<ByteBuf, class_9129>> original) {
		return original.call(registryManager).andThen(registryByteBuf -> {
			FabricRegistryByteBuf fabricRegistryByteBuf = (FabricRegistryByteBuf) registryByteBuf;
			fabricRegistryByteBuf.fabric_setSendableConfigurationChannels(Set.copyOf(addon.getSendableChannels()));
			return registryByteBuf;
		});
	}
}
