/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.object.builder.v1.villager;

import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.mixin.object.builder.VillagerProfessionAccessor;
import net.minecraft.class_1792;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_3852;
import net.minecraft.class_3853;
import net.minecraft.class_3888;
import net.minecraft.class_4158;

/**
 * Allows for the creation of new {@link class_3852}s.
 *
 * <p>The texture for the villagers are located at <code>assets/IDENTIFIER_NAMESPACE/textures/entity/villager/profession/IDENTIFIER_PATH.png</code>
 *
 * <p>A corresponding <code>IDENTIFIER_PATH.mcmeta</code> file exits in the same directory to define properties such as the {@link class_3888.class_3889 HatType} this profession would use.
 *
 * <p>Note this does not register any trades to these villagers. To register trades, add a new entry with your profession as the key to {@link class_3853#field_17067}.
 */
public final class VillagerProfessionBuilder {
	private final ImmutableSet.Builder<class_1792> gatherableItemsBuilder = ImmutableSet.builder();
	private final ImmutableSet.Builder<class_2248> secondaryJobSiteBlockBuilder = ImmutableSet.builder();
	private class_2960 identifier;
	private class_4158 pointOfInterestType;
	@Nullable
	private class_3414 workSoundEvent;

	private VillagerProfessionBuilder() {
	}

	/**
	 * Creates a builder instance to allow for creation of a {@link class_3852}.
	 *
	 * @return A new builder.
	 */
	public static VillagerProfessionBuilder create() {
		return new VillagerProfessionBuilder();
	}

	/**
	 * The Identifier used to identify this villager profession.
	 *
	 * @param id The identifier to assign to this profession.
	 * @return this builder
	 */
	public VillagerProfessionBuilder id(class_2960 id) {
		this.identifier = id;
		return this;
	}

	/**
	 * The {@link class_4158} the Villager of this profession will search for when finding a workstation.
	 *
	 * @param type The {@link class_4158} the Villager will attempt to find.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder workstation(class_4158 type) {
		this.pointOfInterestType = type;
		return this;
	}

	/**
	 * Items that a Villager may harvest in this profession.
	 *
	 * <p>In Vanilla, this is used by the farmer to define what type of crops the farmer can harvest.
	 *
	 * @param items Items harvestable by this profession.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder harvestableItems(class_1792... items) {
		this.gatherableItemsBuilder.add(items);
		return this;
	}

	/**
	 * Items that a Villager may harvest in this profession.
	 *
	 * <p>In Vanilla, this is used by the farmer to define what type of crops the farmer can harvest.
	 *
	 * @param items Items harvestable by this profession.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder harvestableItems(Iterable<class_1792> items) {
		this.gatherableItemsBuilder.addAll(items);
		return this;
	}

	/**
	 * A collection of blocks blocks which may suffice as a secondary job site for a Villager.
	 *
	 * <p>In Vanilla, this is used by the {@link class_3852#field_17056 Farmer} to stay near {@link class_2246#field_10362 Farmland} when at it's job site.
	 *
	 * @param blocks Collection of secondary job site blocks.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder secondaryJobSites(class_2248... blocks) {
		this.secondaryJobSiteBlockBuilder.add(blocks);
		return this;
	}

	/**
	 * A collection of blocks blocks which may suffice as a secondary job site for a Villager.
	 *
	 * <p>In Vanilla, this is used by the {@link class_3852#field_17056 Farmer} to stay near {@link class_2246#field_10362 Farmland} when at it's job site.
	 *
	 * @param blocks Collection of secondary job site blocks.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder secondaryJobSites(Iterable<class_2248> blocks) {
		this.secondaryJobSiteBlockBuilder.addAll(blocks);
		return this;
	}

	/**
	 * Provides the sound made when a Villager works.
	 *
	 * @param workSoundEvent The {@link class_3414} to be played.
	 * @return this builder.
	 */
	public VillagerProfessionBuilder workSound(@Nullable class_3414 workSoundEvent) {
		this.workSoundEvent = workSoundEvent;
		return this;
	}

	/**
	 * Creates the {@link class_3852}.
	 *
	 * @return a new {@link class_3852}.
	 * @throws IllegalStateException if the builder is missing an {@link class_2960 id} and {@link class_4158 workstation}.
	 */
	public class_3852 build() {
		checkState(this.identifier != null, "An Identifier is required to build a new VillagerProfession.");
		checkState(this.pointOfInterestType != null, "A PointOfInterestType is required to build a new VillagerProfession.");
		return VillagerProfessionAccessor.create(this.identifier.toString(), this.pointOfInterestType, this.gatherableItemsBuilder.build(), this.secondaryJobSiteBlockBuilder.build(), this.workSoundEvent);
	}
}
