/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.object.builder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;
import net.fabricmc.fabric.api.object.builder.v1.trade.TradeOfferHelper;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_3852;
import net.minecraft.class_3853;
import net.minecraft.class_5321;

public final class TradeOfferInternals {
	private TradeOfferInternals() {
	}

	/**
	 * Make the rebalanced profession map modifiable, then copy all vanilla
	 * professions' trades to prevent modifications from propagating to the rebalanced one.
	 */
	private static void initVillagerTrades() {
		if (!(class_3853.field_45128 instanceof HashMap)) {
			Map<class_5321<class_3852>, Int2ObjectMap<class_3853.class_1652[]>> map = new HashMap<>(class_3853.field_45128);

			for (Map.Entry<class_5321<class_3852>, Int2ObjectMap<class_3853.class_1652[]>> trade : class_3853.field_17067.entrySet()) {
				if (!map.containsKey(trade.getKey())) map.put(trade.getKey(), trade.getValue());
			}

			class_3853.field_45128 = map;
		}
	}

	// synchronized guards against concurrent modifications - Vanilla does not mutate the underlying arrays (as of 1.16),
	// so reads will be fine without locking.
	public static synchronized void registerVillagerOffers(class_5321<class_3852> profession, int level, TradeOfferHelper.VillagerOffersAdder factory) {
		Objects.requireNonNull(profession, "VillagerProfession may not be null.");
		initVillagerTrades();
		registerOffers(class_3853.field_17067.computeIfAbsent(profession, key -> new Int2ObjectOpenHashMap<>()), level, trades -> factory.onRegister(trades, false));
		registerOffers(class_3853.field_45128.computeIfAbsent(profession, key -> new Int2ObjectOpenHashMap<>()), level, trades -> factory.onRegister(trades, true));
	}

	private static void registerOffers(Int2ObjectMap<class_3853.class_1652[]> leveledTradeMap, int level, Consumer<List<class_3853.class_1652>> factory) {
		final List<class_3853.class_1652> list = new ArrayList<>();
		factory.accept(list);

		final class_3853.class_1652[] originalEntries = leveledTradeMap.computeIfAbsent(level, key -> new class_3853.class_1652[0]);
		final class_3853.class_1652[] addedEntries = list.toArray(new class_3853.class_1652[0]);

		final class_3853.class_1652[] allEntries = ArrayUtils.addAll(originalEntries, addedEntries);
		leveledTradeMap.put(level, allEntries);
	}

	public static class WanderingTraderOffersBuilderImpl implements TradeOfferHelper.WanderingTraderOffersBuilder {
		private static final Object2IntMap<class_2960> ID_TO_INDEX = class_156.method_654(new Object2IntOpenHashMap<>(), idToIndex -> {
			idToIndex.put(BUY_ITEMS_POOL, 0);
			idToIndex.put(SELL_SPECIAL_ITEMS_POOL, 1);
			idToIndex.put(SELL_COMMON_ITEMS_POOL, 2);
		});

		private static final Map<class_2960, class_3853.class_1652[]> DELAYED_MODIFICATIONS = new HashMap<>();

		/**
		 * Make the trade list modifiable.
		 */
		static void initWanderingTraderTrades() {
			if (!(class_3853.field_17724 instanceof ArrayList)) {
				class_3853.field_17724 = new ArrayList<>(class_3853.field_17724);
			}
		}

		@Override
		public TradeOfferHelper.WanderingTraderOffersBuilder pool(class_2960 id, int count, class_3853.class_1652... factories) {
			if (factories.length == 0) throw new IllegalArgumentException("cannot add empty pool");
			if (count <= 0) throw new IllegalArgumentException("count must be positive");

			Objects.requireNonNull(id, "id cannot be null");

			if (ID_TO_INDEX.containsKey(id)) throw new IllegalArgumentException("pool id %s is already registered".formatted(id));

			Pair<class_3853.class_1652[], Integer> pool = Pair.of(factories, count);
			initWanderingTraderTrades();
			ID_TO_INDEX.put(id, class_3853.field_17724.size());
			class_3853.field_17724.add(pool);
			class_3853.class_1652[] delayedModifications = DELAYED_MODIFICATIONS.remove(id);

			if (delayedModifications != null) addOffersToPool(id, delayedModifications);

			return this;
		}

		@Override
		public TradeOfferHelper.WanderingTraderOffersBuilder addOffersToPool(class_2960 pool, class_3853.class_1652... factories) {
			if (!ID_TO_INDEX.containsKey(pool)) {
				DELAYED_MODIFICATIONS.compute(pool, (id, current) -> {
					if (current == null) return factories;

					return ArrayUtils.addAll(current, factories);
				});
				return this;
			}

			int poolIndex = ID_TO_INDEX.getInt(pool);
			initWanderingTraderTrades();
			Pair<class_3853.class_1652[], Integer> poolPair = class_3853.field_17724.get(poolIndex);
			class_3853.class_1652[] modified = ArrayUtils.addAll(poolPair.getLeft(), factories);
			class_3853.field_17724.set(poolIndex, Pair.of(modified, poolPair.getRight()));
			return this;
		}
	}
}
