/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry.sync;

import com.mojang.serialization.Lifecycle;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.api.event.registry.RegistryEntryAddedCallback;
import net.fabricmc.fabric.mixin.registry.sync.AccessorRegistry;
import net.minecraft.class_2378;
import net.minecraft.class_2385;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_5458;

/**
 * Handles synchronising changes to the built-in registries into the dynamic registry manager's template manager,
 * in case it gets classloaded early.
 */
public class DynamicRegistrySync {
	private static final Logger LOGGER = LogManager.getLogger();

	/**
	 * Sets up a synchronisation that will propagate added entries to the given dynamic registry manager, which
	 * should be the <em>built-in</em> manager. It is never destroyed. We don't ever have to unregister
	 * the registry events.
	 */
	public static void setupSync(class_5455.class_5457 template) {
		LOGGER.debug("Setting up synchronisation of new BuiltinRegistries entries to the built-in DynamicRegistryManager");
		class_5458.field_25926.method_10220().forEach(source -> setupSync(source, template));
	}

	/**
	 * Sets up an event registration for the source registy that will ensure all entries added from now on
	 * are also added to the template for dynamic registry managers.
	 */
	private static <T> void setupSync(class_2378<T> source, class_5455.class_5457 template) {
		@SuppressWarnings("unchecked") AccessorRegistry<T> sourceAccessor = (AccessorRegistry<T>) source;
		class_5321<? extends class_2378<T>> sourceKey = source.method_30517();
		class_2385<T> target = (class_2385<T>) template.method_30530(sourceKey);

		RegistryEntryAddedCallback.event(source).register((rawId, id, object) -> {
			LOGGER.trace("Synchronizing {} from built-in registry {} into built-in dynamic registry manager template.",
					id, source.method_30517());
			Lifecycle lifecycle = sourceAccessor.callGetEntryLifecycle(object);
			class_5321<T> entryKey = class_5321.method_29179(sourceKey, id);
			target.method_10273(rawId, entryKey, object, lifecycle);
		});
	}
}
