/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry.sync;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import net.minecraft.class_2378;
import net.minecraft.class_2385;
import net.minecraft.class_2487;
import net.minecraft.class_2507;
import net.minecraft.class_2960;
import net.minecraft.class_5455;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * This solves a bug in vanilla where datapack added biome IDs are not saved to disk. Thus adding or changing a biome
 * from a datapack/mod causes the ids to shift. This remaps the IDs in the {@link class_5455} in a similar
 * manner to the normal registry sync.
 *
 * <p>See: https://bugs.mojang.com/browse/MC-202036
 *
 * <p>This may cause issues when vanilla adds biomes in the future, this should be fixable by also remapping the static ID
 * map vanilla keeps.
 */
public class PersistentDynamicRegistryHandler {
	private static final Logger LOGGER = LogManager.getLogger();

	public static void remapDynamicRegistries(class_5455.class_5457 dynamicRegistryManager, Path saveDir) {
		LOGGER.debug("Starting registry remap");

		class_2487 registryData;

		try {
			registryData = remapDynamicRegistries(dynamicRegistryManager, readCompoundTag(getDataPath(saveDir)));
		} catch (RemapException | IOException e) {
			throw new RuntimeException("Failed to read dynamic registry data", e);
		}

		writeCompoundTag(registryData, getDataPath(saveDir));
	}

	@NotNull
	private static class_2487 remapDynamicRegistries(class_5455.class_5457 dynamicRegistryManager, @Nullable class_2487 existingTag) throws RemapException {
		class_2487 registries = new class_2487();

		// For now we only care about biomes, but lets keep our options open
		class_2487 biomeRegistryData = null;

		if (existingTag != null) {
			biomeRegistryData = existingTag.method_10562(class_2378.field_25114.method_29177().toString());
		}

		class_2385<?> registry = (class_2385<?>) dynamicRegistryManager.method_30530(class_2378.field_25114);
		class_2487 biomeIdMap = remapRegistry(class_2378.field_25114.method_29177(), registry, biomeRegistryData);
		registries.method_10566(class_2378.field_25114.method_29177().toString(), biomeIdMap);

		return registries;
	}

	/**
	 * Remaps a registry if existing data is passed in.
	 * Then writes out the ids in the registry (remapped or a new world).
	 * Keeps hold of the orphaned registry entries as to not overwrite them.
	 */
	private static <T> class_2487 remapRegistry(class_2960 registryId, class_2385<T> registry, @Nullable class_2487 existingTag) throws RemapException {
		if (!(registry instanceof RemappableRegistry)) {
			throw new UnsupportedOperationException("Cannot remap un re-mappable registry: " + registryId.toString());
		}

		// This includes biomes added via datapacks via the vanilla method, along with mod provided biomes.
		boolean isModified = registry.method_10235().stream().anyMatch(id -> !id.method_12836().equals("minecraft"));

		// The current registry might not be modified, but we might have previous changed vanilla ids that we should try and remap
		if (existingTag != null && !isModified) {
			isModified = existingTag.method_10541().stream()
					.map(existingTag::method_10558)
					.map(class_2960::new)
					.anyMatch(id -> !id.method_12836().equals("minecraft"));
		}

		if (LOGGER.isDebugEnabled()) {
			if (existingTag == null) {
				LOGGER.debug("No existing data found, assuming new registry with {} entries. modded = {}", registry.method_10235().size(), isModified);
			} else {
				LOGGER.debug("Existing registry data found. modded = {}", isModified);

				for (T entry : registry) {
					//noinspection unchecked
					class_2960 id = registry.method_10221(entry);
					int rawId = registry.method_10206(entry);

					if (id == null || rawId < 0) continue;

					if (existingTag.method_10541().contains(id.toString())) {
						int existingRawId = existingTag.method_10550(id.toString());

						if (rawId != existingRawId) {
							LOGGER.debug("Remapping {} {} -> {}", id.toString(), rawId, existingRawId);
						} else {
							LOGGER.debug("Using existing id for {} {}", id.toString(), rawId);
						}
					} else {
						LOGGER.debug("Found new registry entry {}", id.toString());
					}
				}
			}
		}

		// If we have some existing ids and the registry contains modded/datapack entries we remap the registry with those
		if (existingTag != null && isModified) {
			LOGGER.debug("Remapping {} with {} entries", registryId, registry.method_10235().size());
			Object2IntMap<class_2960> idMap = new Object2IntOpenHashMap<>();

			for (String key : existingTag.method_10541()) {
				idMap.put(new class_2960(key), existingTag.method_10550(key));
			}

			((RemappableRegistry) registry).remap(registryId.toString(), idMap, RemappableRegistry.RemapMode.AUTHORITATIVE);
		} else {
			LOGGER.debug("Skipping remap of {}", registryId);
		}

		// Now start to build up what we are going to save out
		class_2487 registryTag = new class_2487();

		// Save all ids as they appear in the remapped, or new registry to disk even if not modded.
		for (T entry : registry) {
			//noinspection unchecked
			class_2960 id = registry.method_10221(entry);

			if (id == null) {
				continue;
			}

			int rawId = registry.method_10206(entry);
			registryTag.method_10569(id.toString(), rawId);
		}

		/*
		 * Look for existing registry key/values that are not in the current registries.
		 * This can happen when registry entries are removed, preventing that ID from being re-used by something else.
		 */
		if (existingTag != null) {
			for (String key : existingTag.method_10541()) {
				if (!registryTag.method_10545(key)) {
					LOGGER.debug("Saving orphaned registry entry: " + key);
					registryTag.method_10569(key, registryTag.method_10550(key));
				}
			}
		}

		return registryTag;
	}

	private static Path getDataPath(Path saveDir) {
		return saveDir.resolve("data").resolve("fabricDynamicRegistry.dat");
	}

	@Nullable
	private static class_2487 readCompoundTag(Path path) throws IOException {
		if (!Files.exists(path)) {
			return null;
		}

		try (InputStream inputStream = Files.newInputStream(path)) {
			class_2487 compoundTag = class_2507.method_10629(inputStream);

			if (!compoundTag.method_10545("version") || !compoundTag.method_10545("registries") || compoundTag.method_10550("version") != 1) {
				throw new UnsupportedOperationException("Unsupported dynamic registry data format. Try updating?");
			}

			return compoundTag.method_10562("registries");
		}
	}

	private static void writeCompoundTag(class_2487 compoundTag, Path path) {
		try {
			Files.createDirectories(path.getParent());

			try (OutputStream outputStream = Files.newOutputStream(path, StandardOpenOption.CREATE)) {
				class_2487 outputTag = new class_2487();
				outputTag.method_10569("version", 1);
				outputTag.method_10566("registries", compoundTag);

				class_2507.method_10634(outputTag, outputStream);
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}
}
