/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry.sync;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.fabricmc.fabric.api.event.registry.RegistryAttributeHolder;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1255;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2596;
import net.minecraft.class_2960;

public final class RegistrySyncManager {
	static final boolean DEBUG = System.getProperty("fabric.registry.debug", "false").equalsIgnoreCase("true");
	static final class_2960 ID = new class_2960("fabric", "registry/sync");
	private static final Logger LOGGER = LogManager.getLogger("FabricRegistrySync");
	private static final boolean DEBUG_WRITE_REGISTRY_DATA = System.getProperty("fabric.registry.debug.writeContentsAsCsv", "false").equalsIgnoreCase("true");

	//Set to true after vanilla's bootstrap has completed
	public static boolean postBootstrap = false;

	private RegistrySyncManager() { }

	public static class_2596<?> createPacket() {
		LOGGER.debug("Creating registry sync packet");

		class_2487 tag = toTag(true, null);

		if (tag == null) {
			return null;
		}

		class_2540 buf = new class_2540(Unpooled.buffer());
		buf.method_10794(tag);

		return ServerPlayNetworking.createS2CPacket(ID, buf);
	}

	public static void receivePacket(class_1255<?> executor, class_2540 buf, boolean accept, Consumer<Exception> errorHandler) {
		class_2487 compound = buf.method_10798();

		if (accept) {
			try {
				executor.method_5385(() -> {
					if (compound == null) {
						errorHandler.accept(new RemapException("Received null compound tag in sync packet!"));
						return null;
					}

					try {
						apply(compound, RemappableRegistry.RemapMode.REMOTE);
					} catch (RemapException e) {
						errorHandler.accept(e);
					}

					return null;
				}).get(30, TimeUnit.SECONDS);
			} catch (ExecutionException | InterruptedException | TimeoutException e) {
				errorHandler.accept(e);
			}
		}
	}

	/**
	 * Creates a {@link class_2487} used to save or sync the registry ids.
	 *
	 * @param isClientSync true when syncing to the client, false when saving
	 * @param activeTag contains the registry ids that were previously read and applied, can be null.
	 * @return a {@link class_2487} to save or sync, null when empty
	 */
	@Nullable
	public static class_2487 toTag(boolean isClientSync, @Nullable class_2487 activeTag) {
		class_2487 mainTag = new class_2487();

		for (class_2960 registryId : class_2378.field_11144.method_10235()) {
			class_2378 registry = class_2378.field_11144.method_10223(registryId);

			if (DEBUG_WRITE_REGISTRY_DATA) {
				File location = new File(".fabric" + File.separatorChar + "debug" + File.separatorChar + "registry");
				boolean c = true;

				if (!location.exists()) {
					if (!location.mkdirs()) {
						LOGGER.warn("[fabric-registry-sync debug] Could not create " + location.getAbsolutePath() + " directory!");
						c = false;
					}
				}

				if (c && registry != null) {
					File file = new File(location, registryId.toString().replace(':', '.').replace('/', '.') + ".csv");

					try (FileOutputStream stream = new FileOutputStream(file)) {
						StringBuilder builder = new StringBuilder("Raw ID,String ID,Class Type\n");

						for (Object o : registry) {
							String classType = (o == null) ? "null" : o.getClass().getName();
							//noinspection unchecked
							class_2960 id = registry.method_10221(o);
							if (id == null) continue;

							//noinspection unchecked
							int rawId = registry.method_10206(o);
							String stringId = id.toString();
							builder.append("\"").append(rawId).append("\",\"").append(stringId).append("\",\"").append(classType).append("\"\n");
						}

						stream.write(builder.toString().getBytes(StandardCharsets.UTF_8));
					} catch (IOException e) {
						LOGGER.warn("[fabric-registry-sync debug] Could not write to " + file.getAbsolutePath() + "!", e);
					}
				}
			}

			/*
			 * This contains the previous state's registry data, this is used for a few things:
			 * Such as ensuring that previously modded registries or registry entries are not lost or overwritten.
			 */
			class_2487 previousRegistryData = null;

			if (activeTag != null && activeTag.method_10545(registryId.toString())) {
				previousRegistryData = activeTag.method_10562(registryId.toString());
			}

			RegistryAttributeHolder attributeHolder = RegistryAttributeHolder.get(registry);

			if (!attributeHolder.hasAttribute(isClientSync ? RegistryAttribute.SYNCED : RegistryAttribute.PERSISTED)) {
				LOGGER.debug("Not {} registry: {}", isClientSync ? "syncing" : "saving", registryId);
				continue;
			}

			/*
			 * Dont do anything with vanilla registries on client sync.
			 * When saving skip none modded registries that doesnt have previous registry data
			 *
			 * This will not sync IDs if a world has been previously modded, either from removed mods
			 * or a previous version of fabric registry sync, but will save these ids to disk in case the mod or mods
			 * are added back.
			 */
			if ((previousRegistryData == null || isClientSync) && !attributeHolder.hasAttribute(RegistryAttribute.MODDED)) {
				LOGGER.debug("Skipping un-modded registry: " + registryId);
				continue;
			} else if (previousRegistryData != null) {
				LOGGER.debug("Preserving previously modded registry: " + registryId);
			}

			if (isClientSync) {
				LOGGER.debug("Syncing registry: " + registryId);
			} else {
				LOGGER.debug("Saving registry: " + registryId);
			}

			if (registry instanceof RemappableRegistry) {
				class_2487 registryTag = new class_2487();
				IntSet rawIdsFound = DEBUG ? new IntOpenHashSet() : null;

				for (Object o : registry) {
					//noinspection unchecked
					class_2960 id = registry.method_10221(o);
					if (id == null) continue;

					//noinspection unchecked
					int rawId = registry.method_10206(o);

					if (DEBUG) {
						if (registry.method_10223(id) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> string ID " + id + " -> object " + registry.method_10223(id) + "!");
						}

						if (registry.method_10200(rawId) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> integer ID " + rawId + " -> object " + registry.method_10200(rawId) + "!");
						}

						if (!rawIdsFound.add(rawId)) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": multiple objects hold the raw ID " + rawId + " (this one is " + id + ")");
						}
					}

					registryTag.method_10569(id.toString(), rawId);
				}

				/*
				 * Look for existing registry key/values that are not in the current registries.
				 * This can happen when registry entries are removed, preventing that ID from being re-used by something else.
				 */
				if (!isClientSync && previousRegistryData != null) {
					for (String key : previousRegistryData.method_10541()) {
						if (!registryTag.method_10545(key)) {
							LOGGER.debug("Saving orphaned registry entry: " + key);
							registryTag.method_10569(key, registryTag.method_10550(key));
						}
					}
				}

				mainTag.method_10566(registryId.toString(), registryTag);
			}
		}

		// Ensure any orphaned registry's are kept on disk
		if (!isClientSync && activeTag != null) {
			for (String registryKey : activeTag.method_10541()) {
				if (!mainTag.method_10545(registryKey)) {
					LOGGER.debug("Saving orphaned registry: " + registryKey);
					mainTag.method_10566(registryKey, activeTag.method_10562(registryKey));
				}
			}
		}

		if (mainTag.method_10541().isEmpty()) {
			return null;
		}

		class_2487 tag = new class_2487();
		tag.method_10569("version", 1);
		tag.method_10566("registries", mainTag);

		return tag;
	}

	public static class_2487 apply(class_2487 tag, RemappableRegistry.RemapMode mode) throws RemapException {
		class_2487 mainTag = tag.method_10562("registries");
		Set<String> containedRegistries = Sets.newHashSet(mainTag.method_10541());

		for (class_2960 registryId : class_2378.field_11144.method_10235()) {
			if (!containedRegistries.remove(registryId.toString())) {
				continue;
			}

			class_2487 registryTag = mainTag.method_10562(registryId.toString());
			class_2378 registry = class_2378.field_11144.method_10223(registryId);

			RegistryAttributeHolder attributeHolder = RegistryAttributeHolder.get(registry);

			if (!attributeHolder.hasAttribute(RegistryAttribute.MODDED)) {
				LOGGER.debug("Not applying registry data to vanilla registry {}", registryId.toString());
				continue;
			}

			if (registry instanceof RemappableRegistry) {
				Object2IntMap<class_2960> idMap = new Object2IntOpenHashMap<>();

				for (String key : registryTag.method_10541()) {
					idMap.put(new class_2960(key), registryTag.method_10550(key));
				}

				((RemappableRegistry) registry).remap(registryId.toString(), idMap, mode);
			}
		}

		if (!containedRegistries.isEmpty()) {
			LOGGER.warn("[fabric-registry-sync] Could not find the following registries: " + Joiner.on(", ").join(containedRegistries));
		}

		return mainTag;
	}

	public static void unmap() throws RemapException {
		for (class_2960 registryId : class_2378.field_11144.method_10235()) {
			class_2378 registry = class_2378.field_11144.method_10223(registryId);

			if (registry instanceof RemappableRegistry) {
				((RemappableRegistry) registry).unmap(registryId.toString());
			}
		}
	}

	public static void bootstrapRegistries() {
		postBootstrap = true;
	}
}
