/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.registry.sync.client;

import org.slf4j.Logger;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.Identifier;
import net.minecraft.world.item.CreativeModeTabs;

import net.fabricmc.fabric.impl.registry.sync.RemapException;
import net.fabricmc.fabric.impl.registry.sync.RemappableRegistry;
import net.fabricmc.fabric.impl.registry.sync.trackers.vanilla.BlockInitTracker;

@Mixin(Minecraft.class)
public class MinecraftMixin {
	@Shadow
	@Final
	private static Logger LOGGER;

	// Unmap the registry before loading a new SP/MP setup.
	@Inject(at = @At("RETURN"), method = "disconnect(Lnet/minecraft/client/gui/screens/Screen;Z)V")
	public void disconnectAfter(Screen disconnectionScreen, boolean bl, CallbackInfo ci) {
		try {
			unmap();
		} catch (RemapException e) {
			LOGGER.warn("Failed to unmap Fabric registries!", e);
		}
	}

	@Inject(method = "<init>", at = @At(value = "INVOKE", target = "Ljava/lang/Thread;currentThread()Ljava/lang/Thread;"))
	private void afterModInit(CallbackInfo ci) {
		// Freeze the registries on the client
		LOGGER.debug("Freezing registries");
		BuiltInRegistries.bootStrap();
		BlockInitTracker.postFreeze();
		CreativeModeTabs.validate();
	}

	@Unique
	private static void unmap() throws RemapException {
		for (Identifier registryId : BuiltInRegistries.REGISTRY.keySet()) {
			Registry<?> registry = BuiltInRegistries.REGISTRY.getValue(registryId);

			if (registry instanceof RemappableRegistry) {
				((RemappableRegistry) registry).unmap();
			}
		}
	}
}
