/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.client.render.fluid.v1;

import java.util.Objects;
import net.minecraft.class_1058;
import net.minecraft.class_1059;
import net.minecraft.class_1920;
import net.minecraft.class_2338;
import net.minecraft.class_2960;
import net.minecraft.class_3610;
import org.jspecify.annotations.Nullable;

/**
 * A simple fluid render handler that uses and loads sprites given by their
 * identifiers. Most fluids don't need more than this. In fact, if a fluid just
 * needs the vanilla water texture with a custom color, {@link #coloredWater}
 * can be used to easily create a fluid render handler for that.
 *
 * <p>Note that it's assumed that the fluid textures are assumed to be
 * registered to the blocks sprite atlas. If they are not, you have to manually
 * register the fluid textures. The "fabric-textures" API may come in handy for
 * that.
 */
public class SimpleFluidRenderHandler implements FluidRenderHandler {
	/**
	 * The vanilla still water texture identifier.
	 */
	public static final class_2960 WATER_STILL = class_2960.method_60656("block/water_still");

	/**
	 * The vanilla flowing water texture identifier.
	 */
	public static final class_2960 WATER_FLOWING = class_2960.method_60656("block/water_flow");

	/**
	 * The vanilla water overlay texture identifier.
	 */
	public static final class_2960 WATER_OVERLAY = class_2960.method_60656("block/water_overlay");

	/**
	 * The vanilla still lava texture identifier.
	 */
	public static final class_2960 LAVA_STILL = class_2960.method_60656("block/lava_still");

	/**
	 * The vanilla flowing lava texture identifier.
	 */
	public static final class_2960 LAVA_FLOWING = class_2960.method_60656("block/lava_flow");

	protected final class_2960 stillTexture;
	protected final class_2960 flowingTexture;
	protected final class_2960 overlayTexture;

	protected final class_1058[] sprites;

	protected final int tint;

	/**
	 * Creates a fluid render handler with an overlay texture and a custom,
	 * fixed tint.
	 *
	 * @param stillTexture The texture for still fluid.
	 * @param flowingTexture The texture for flowing/falling fluid.
	 * @param overlayTexture The texture behind glass, leaves and other
	 * {@linkplain FluidRenderHandlerRegistry#setBlockTransparency registered
	 * transparent blocks}.
	 * @param tint The fluid color RGB. Alpha is ignored.
	 */
	public SimpleFluidRenderHandler(class_2960 stillTexture, class_2960 flowingTexture, @Nullable class_2960 overlayTexture, int tint) {
		this.stillTexture = Objects.requireNonNull(stillTexture, "stillTexture");
		this.flowingTexture = Objects.requireNonNull(flowingTexture, "flowingTexture");
		this.overlayTexture = overlayTexture;
		this.sprites = new class_1058[overlayTexture == null ? 2 : 3];
		this.tint = tint;
	}

	/**
	 * Creates a fluid render handler with an overlay texture and no tint.
	 *
	 * @param stillTexture The texture for still fluid.
	 * @param flowingTexture The texture for flowing/falling fluid.
	 * @param overlayTexture The texture behind glass, leaves and other
	 * {@linkplain FluidRenderHandlerRegistry#setBlockTransparency registered
	 * transparent blocks}.
	 */
	public SimpleFluidRenderHandler(class_2960 stillTexture, class_2960 flowingTexture, class_2960 overlayTexture) {
		this(stillTexture, flowingTexture, overlayTexture, -1);
	}

	/**
	 * Creates a fluid render handler without an overlay texture and a custom,
	 * fixed tint.
	 *
	 * @param stillTexture The texture for still fluid.
	 * @param flowingTexture The texture for flowing/falling fluid.
	 * @param tint The fluid color RGB. Alpha is ignored.
	 */
	public SimpleFluidRenderHandler(class_2960 stillTexture, class_2960 flowingTexture, int tint) {
		this(stillTexture, flowingTexture, null, tint);
	}

	/**
	 * Creates a fluid render handler without an overlay texture and no tint.
	 *
	 * @param stillTexture The texture for still fluid.
	 * @param flowingTexture The texture for flowing/falling fluid.
	 */
	public SimpleFluidRenderHandler(class_2960 stillTexture, class_2960 flowingTexture) {
		this(stillTexture, flowingTexture, null, -1);
	}

	/**
	 * Creates a fluid render handler that uses the vanilla water texture with a
	 * fixed, custom color.
	 *
	 * @param tint The fluid color RGB. Alpha is ignored.
	 * @see	#WATER_STILL
	 * @see	#WATER_FLOWING
	 * @see #WATER_OVERLAY
	 */
	public static SimpleFluidRenderHandler coloredWater(int tint) {
		return new SimpleFluidRenderHandler(WATER_STILL, WATER_FLOWING, WATER_OVERLAY, tint);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public class_1058[] getFluidSprites(@Nullable class_1920 view, @Nullable class_2338 pos, class_3610 state) {
		return sprites;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reloadTextures(class_1059 textureAtlas) {
		sprites[0] = textureAtlas.method_4608(stillTexture);
		sprites[1] = textureAtlas.method_4608(flowingTexture);

		if (overlayTexture != null) {
			sprites[2] = textureAtlas.method_4608(overlayTexture);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getFluidColor(@Nullable class_1920 view, @Nullable class_2338 pos, class_3610 state) {
		return tint;
	}
}
