/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.tag;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.Lifecycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4080;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_6862;
import net.minecraft.class_7225;
import net.minecraft.class_7780;

public final class TagAliasLoader extends class_4080<Map<class_5321<? extends class_2378<?>>, List<TagAliasLoader.Data>>> implements IdentifiableResourceReloadListener {
	public static final class_2960 ID = class_2960.method_60655("fabric-tag-api-v1", "tag_alias_groups");

	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-tag-api-v1");
	private final class_7225.class_7874 registries;

	public TagAliasLoader(class_7225.class_7874 registries) {
		this.registries = registries;
	}

	@Override
	public class_2960 getFabricId() {
		return ID;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected Map<class_5321<? extends class_2378<?>>, List<TagAliasLoader.Data>> method_18789(class_3300 manager, class_3695 profiler) {
		Map<RegistryKey<? extends Registry<?>>, List<TagAliasLoader.Data>> dataByRegistry = new HashMap<>();
		Iterator<RegistryKey<? extends Registry<?>>> registryIterator = registries.streamAllRegistryKeys().iterator();

		while (registryIterator.hasNext()) {
			RegistryKey<? extends Registry<?>> registryKey = registryIterator.next();
			ResourceFinder resourceFinder = ResourceFinder.json(getDirectory(registryKey));

			for (Map.Entry<Identifier, Resource> entry : resourceFinder.findResources(manager).entrySet()) {
				Identifier resourcePath = entry.getKey();
				Identifier groupId = resourceFinder.toResourceId(resourcePath);

				try (Reader reader = entry.getValue().getReader()) {
					JsonElement json = StrictJsonParser.parse(reader);
					Codec<TagAliasGroup<Object>> codec = TagAliasGroup.codec((RegistryKey<? extends Registry<Object>>) registryKey);

					switch (codec.parse(JsonOps.INSTANCE, json)) {
					case DataResult.Success(TagAliasGroup<Object> group, Lifecycle unused) -> {
						var data = new Data(groupId, group);
						dataByRegistry.computeIfAbsent(registryKey, key -> new ArrayList<>()).add(data);
					}
					case DataResult.Error<?> error -> {
						LOGGER.error("[Fabric] Couldn't parse tag alias group file '{}' from '{}': {}", groupId, resourcePath, error.message());
					}
					}
				} catch (IOException | JsonParseException e) {
					LOGGER.error("[Fabric] Couldn't parse tag alias group file '{}' from '{}'", groupId, resourcePath, e);
				}
			}
		}

		return dataByRegistry;
	}

	private static String getDirectory(class_5321<? extends class_2378<?>> registryKey) {
		String directory = "fabric/tag_alias/";
		class_2960 registryId = registryKey.method_29177();

		if (!class_2960.field_33381.equals(registryId.method_12836())) {
			directory += registryId.method_12836() + '/';
		}

		return directory + registryId.method_12832();
	}

	@Override
	protected void apply(Map<class_5321<? extends class_2378<?>>, List<TagAliasLoader.Data>> prepared, class_3300 manager, class_3695 profiler) {
		for (Map.Entry<class_5321<? extends class_2378<?>>, List<Data>> entry : prepared.entrySet()) {
			Map<class_6862<?>, Set<class_6862<?>>> groupsByTag = new HashMap<>();

			for (Data data : entry.getValue()) {
				Set<class_6862<?>> group = new HashSet<>(data.group.tags());

				for (class_6862<?> tag : data.group.tags()) {
					Set<class_6862<?>> oldGroup = groupsByTag.get(tag);

					// If there's an old group...
					if (oldGroup != null) {
						// ...merge all of its tags into the current group...
						group.addAll(oldGroup);

						// ...and replace the recorded group of each tag in the old group with the new group.
						for (class_6862<?> other : oldGroup) {
							groupsByTag.put(other, group);
						}
					}

					groupsByTag.put(tag, group);
				}
			}

			// Remove any groups of one tag, we don't need to apply them.
			groupsByTag.values().removeIf(tags -> tags.size() == 1);

			class_7225.class_7226<?> wrapper = registries.method_46762(entry.getKey());

			if (wrapper instanceof TagAliasEnabledRegistryWrapper aliasWrapper) {
				aliasWrapper.fabric_loadTagAliases(groupsByTag);
			} else {
				throw new ClassCastException("[Fabric] Couldn't apply tag aliases to registry wrapper %s (%s) since it doesn't implement TagAliasEnabledRegistryWrapper"
						.formatted(wrapper, entry.getKey().method_29177()));
			}
		}
	}

	public static <T> void applyToDynamicRegistries(class_7780<T> registries, T phase) {
		Iterator<class_5455.class_6892<?>> registryEntries = registries.method_45928(phase).method_40311().iterator();

		while (registryEntries.hasNext()) {
			class_2378<?> registry = registryEntries.next().comp_351();

			if (registry instanceof SimpleRegistryExtension extension) {
				extension.fabric_applyPendingTagAliases();
				// This is not needed in the static registry code path as the tag aliases are applied
				// before the tags are refreshed. Dynamic registry loading (including tags) takes place earlier
				// than the rest of a data reload, so we need to refresh the tags manually.
				extension.fabric_refreshTags();
			} else {
				throw new ClassCastException("[Fabric] Couldn't apply pending tag aliases to registry %s (%s) since it doesn't implement SimpleRegistryExtension"
						.formatted(registry, registry.getClass().getName()));
			}
		}
	}

	protected record Data(class_2960 groupId, TagAliasGroup<?> group) {
	}
}
