/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.fluid;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidConstants;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.StoragePreconditions;
import net.fabricmc.fabric.api.transfer.v1.storage.base.ExtractionOnlyStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.base.SingleSlotStorage;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1844;
import net.minecraft.class_1847;
import net.minecraft.class_3612;
import net.minecraft.class_9334;

/**
 * Implementation of the storage for a water potion.
 */
public class WaterPotionStorage implements ExtractionOnlyStorage<FluidVariant>, SingleSlotStorage<FluidVariant> {
	private static final FluidVariant CONTAINED_FLUID = FluidVariant.of(class_3612.field_15910);
	private static final long CONTAINED_AMOUNT = FluidConstants.BOTTLE;

	@Nullable
	public static WaterPotionStorage find(ContainerItemContext context) {
		return isWaterPotion(context) ? new WaterPotionStorage(context) : null;
	}

	private static boolean isWaterPotion(ContainerItemContext context) {
		ItemVariant variant = context.getItemVariant();
		class_1844 potionContents = variant.getComponentMap()
				.method_58695(class_9334.field_49651, class_1844.field_49274);
		return variant.isOf(class_1802.field_8574) && potionContents.comp_2378().orElse(null) == class_1847.field_8991;
	}

	private final ContainerItemContext context;

	private WaterPotionStorage(ContainerItemContext context) {
		this.context = context;
	}

	private boolean isWaterPotion() {
		return isWaterPotion(context);
	}

	private ItemVariant mapToGlassBottle() {
		class_1799 newStack = context.getItemVariant().toStack();
		newStack.method_57379(class_9334.field_49651, class_1844.field_49274);
		return ItemVariant.of(class_1802.field_8469, newStack.method_57380());
	}

	@Override
	public long extract(FluidVariant resource, long maxAmount, TransactionContext transaction) {
		StoragePreconditions.notBlankNotNegative(resource, maxAmount);

		// Not a water potion anymore
		if (!isWaterPotion()) return 0;

		// Make sure that the fluid and the amount match.
		if (resource.equals(CONTAINED_FLUID) && maxAmount >= CONTAINED_AMOUNT) {
			if (context.exchange(mapToGlassBottle(), 1, transaction) == 1) {
				// Conversion ok!
				return CONTAINED_AMOUNT;
			}
		}

		return 0;
	}

	@Override
	public boolean isResourceBlank() {
		return getResource().isBlank();
	}

	@Override
	public FluidVariant getResource() {
		// Only contains a resource if this is still a water potion.
		if (isWaterPotion()) {
			return CONTAINED_FLUID;
		} else {
			return FluidVariant.blank();
		}
	}

	@Override
	public long getAmount() {
		if (isWaterPotion()) {
			return CONTAINED_AMOUNT;
		} else {
			return 0;
		}
	}

	@Override
	public long getCapacity() {
		// Capacity is the same as the amount.
		return getAmount();
	}

	@Override
	public String toString() {
		return "WaterPotionStorage[" + context + "]";
	}
}
