/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.content.registry;

import java.util.IdentityHashMap;
import java.util.Map;

import it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.api.event.lifecycle.v1.ServerLifecycleEvents;
import net.fabricmc.fabric.api.registry.FuelRegistry;
import net.minecraft.class_1792;
import net.minecraft.class_1935;
import net.minecraft.class_2609;
import net.minecraft.class_3494;

// TODO: Clamp values to 32767 (+ add hook for mods which extend the limit to disable the check?)
public final class FuelRegistryImpl implements FuelRegistry {
	private static final Logger LOGGER = LogManager.getLogger();
	private final Object2IntMap<class_1935> itemCookTimes = new Object2IntLinkedOpenHashMap<>();
	private final Object2IntMap<class_3494<class_1792>> tagCookTimes = new Object2IntLinkedOpenHashMap<>();
	private volatile Map<class_1792, Integer> fuelTimeCache = null; // thread safe via copy-on-write mechanism

	public FuelRegistryImpl() {
		ServerLifecycleEvents.END_DATA_PACK_RELOAD.register((server, serverResourceManager, success) -> {
			if (success) {
				resetCache();
			}
		});
	}

	public Map<class_1792, Integer> getFuelTimes() {
		Map<class_1792, Integer> ret = fuelTimeCache;

		if (ret == null) {
			fuelTimeCache = ret = new IdentityHashMap<>(class_2609.method_11196()); // IdentityHashMap is faster than vanilla's LinkedHashMap and suitable for Item keys
		}

		return ret;
	}

	@Override
	public Integer get(class_1935 item) {
		return getFuelTimes().get(item.method_8389());
	}

	@Override
	public void add(class_1935 item, Integer cookTime) {
		if (cookTime > 32767) {
			LOGGER.warn("Tried to register an overly high cookTime: " + cookTime + " > 32767! (" + item + ")");
		}

		itemCookTimes.put(item, cookTime.intValue());
		resetCache();
	}

	@Override
	public void add(class_3494<class_1792> tag, Integer cookTime) {
		if (cookTime > 32767) {
			LOGGER.warn("Tried to register an overly high cookTime: " + cookTime + " > 32767! (" + getTagName(tag) + ")");
		}

		tagCookTimes.put(tag, cookTime.intValue());
		resetCache();
	}

	@Override
	public void remove(class_1935 item) {
		add(item, 0);
		resetCache();
	}

	@Override
	public void remove(class_3494<class_1792> tag) {
		add(tag, 0);
		resetCache();
	}

	@Override
	public void clear(class_1935 item) {
		itemCookTimes.removeInt(item);
		resetCache();
	}

	@Override
	public void clear(class_3494<class_1792> tag) {
		tagCookTimes.removeInt(tag);
		resetCache();
	}

	public void apply(Map<class_1792, Integer> map) {
		// tags take precedence before blocks
		for (class_3494<class_1792> tag : tagCookTimes.keySet()) {
			int time = tagCookTimes.getInt(tag);

			if (time <= 0) {
				for (class_1792 i : tag.method_15138()) {
					map.remove(i);
				}
			} else {
				class_2609.method_11194(map, tag, time);
			}
		}

		for (class_1935 item : itemCookTimes.keySet()) {
			int time = itemCookTimes.getInt(item);

			if (time <= 0) {
				map.remove(item.method_8389());
			} else {
				class_2609.method_11202(map, item, time);
			}
		}
	}

	private static String getTagName(class_3494<?> tag) {
		if (tag instanceof class_3494.class_5123) {
			return ((class_3494.class_5123<?>) tag).method_26791().toString();
		}

		return tag.toString();
	}

	public void resetCache() {
		fuelTimeCache = null;
	}
}
